#!/bin/bash

# This script converts a DirectAdmin DNS zone file to a cPanel-compatible format.
#
# Usage: ./convert_da_to_cpanel.sh [--force] <directadmin_zone_file>
#   --force  Reconvert even if the file already appears to be in cPanel format
#
# The script will modify the file in-place.

set -e

FORCE=0
if [ "${1-}" = "--force" ]; then
    FORCE=1
    shift
fi

# --- Configuration ---
# You can change the default TTL and mail priority here if needed.
DEFAULT_TTL="14400"
MX_PRIORITY="0"

# --- Script ---

if [ -z "$1" ]; then
    echo "Usage: $0 [--force] <directadmin_zone_file>"
    exit 1
fi

DA_FILE=$1

if [ ! -f "$DA_FILE" ]; then
    echo "Error: File not found: $DA_FILE"
    exit 1
fi

# Check if file is already in cPanel format (unless forced)
if [ "$FORCE" -ne 1 ] && grep -q "^; cPanel first:" "$DA_FILE"; then
    echo "File $DA_FILE is already in cPanel format. Skipping conversion."
    exit 0
fi

# Try to get domain from NS record
DOMAIN_FROM_NS=$(grep -E "\s+IN\s+NS\s+" "$DA_FILE" | awk '{print $1}' | head -n 1 | sed 's/\.$//')
if [ -n "$DOMAIN_FROM_NS" ] && [ "$DOMAIN_FROM_NS" != "@" ]; then
    DOMAIN=$DOMAIN_FROM_NS
else
    DOMAIN=$(basename "$DA_FILE" .db)
fi

TMP_FILE=$(mktemp)

# Get the main IP address for the domain from its A record.
MAIN_IP=$(grep -E "^(${DOMAIN}\.|@)\s+.*\s+IN\s+A\s+" "$DA_FILE" | awk '{print $NF}' | head -n 1 || true)

if [ -z "$MAIN_IP" ]; then
    echo "Warning: Could not determine main IP for ${DOMAIN}. CNAME conversion for subdomains might be affected."
fi

# Get the list of nameservers.
NAMESERVERS=$(grep -E "\s+IN\s+NS\s+" "$DA_FILE" | awk '{print $NF}' | tr '\n' ' ')
PRIMARY_NS=$(echo "$NAMESERVERS" | cut -d' ' -f1)
if [ -z "$PRIMARY_NS" ]; then
    echo "Warning: Could not determine primary nameserver for ${DOMAIN}."
    PRIMARY_NS="ns1.${DOMAIN}."
fi


# --- File Conversion ---

# Create the cPanel header with a current timestamp.
UPDATE_TIME=$(date +%s)
SERIAL=$(date +%Y%m%d)01
ADMIN_EMAIL="hostmaster.${DOMAIN}."

cat > "$TMP_FILE" << EOL
; cPanel first:130.0.5 (update_time):${UPDATE_TIME} 130.0.5:1 Cpanel::ZoneFile::VERSION:1.3
; Zone file for ${DOMAIN}
\$TTL ${DEFAULT_TTL}
@      ${DEFAULT_TTL}    IN      SOA     ${PRIMARY_NS} ${ADMIN_EMAIL} (
                ${SERIAL}      ; serial, todays date+todays
                3600            ; refresh, seconds
                1800            ; retry, seconds
                1209600         ; expire, seconds
                86400 )         ; minimum, seconds

EOL

# Process the records using awk, skipping the old header.
awk -v domain="${DOMAIN}" \
    -v main_ip="${MAIN_IP}" \
    -v nameservers_str="${NAMESERVERS}" \
    -v mx_priority="${MX_PRIORITY}" \
    -v default_ttl="${DEFAULT_TTL}" '
BEGIN {
    split(nameservers_str, ns_array, " ");
    for (i in ns_array) {
        if (ns_array[i] != "") {
            ns_map[ns_array[i]] = 1;
        }
    }
    mx_printed = 0;
    last_record_type = "";
}

# Skip comments, blank lines, TTL, and SOA records as we have created our own.
/^\s*;/ || /^\$TTL/ { next }
/IN\s+SOA/ {
    in_soa = 1
}
in_soa {
    if (/\)/) {
        in_soa = 0
    }
    next
}

{
    # Skip blank lines
    if ($0 ~ /^\s*$/) {
        next
    }
    record_name = $1
    
    # Handle records with and without TTL specified.
    if ($3 == "IN") { # TTL is present
        record_type = $4
        value_start_index = 5
    } else if ($2 == "IN") { # TTL is not present
        record_type = $3
        value_start_index = 4
    } else {
        next # Skip lines that are not DNS records.
    }

    # Reconstruct the value part of the record.
    value = ""
    for (i = value_start_index; i <= NF; i++) {
        value = value $i " "
    }
    sub(/\s+$/, "", value)

    if (record_name == "@") {
        record_name = domain ".";
    }

    # Convert A records for common subdomains to CNAMEs if they share the main IP.
    if (record_type == "A") {
        ip = $NF;
        is_ns = 0;
        for (ns in ns_map) {
            if (record_name == ns) {
                is_ns = 1;
                break;
            }
        }

        if (is_ns) {
            # Add spacing between different record types (only for major type changes)
            if ((last_record_type == "MX" && record_type != "MX") ||
                (last_record_type == "CNAME" && record_type != "CNAME") ||
                (last_record_type == "NS" && record_type != "NS") ||
                (last_record_type == "TXT" && record_type != "TXT")) {
                print "";
            }
            printf "%-24s %s IN A %s\n", record_name, default_ttl, ip;
            last_record_type = "A";
        } else if (main_ip != "" && ip == main_ip && record_name != domain ".") {
            # Add spacing between different record types (only for major type changes)
            if ((last_record_type == "A" && record_type != "A") || 
                (last_record_type == "MX" && record_type != "MX") ||
                (last_record_type == "NS" && record_type != "NS") ||
                (last_record_type == "TXT" && record_type != "TXT")) {
                    print "";
            }
            printf "%-24s %s IN CNAME %s\n", record_name, default_ttl, domain ".";
            last_record_type = "CNAME";
        } else {
            # Add spacing between different record types (only for major type changes)
            if ((last_record_type == "MX" && record_type != "MX") ||
                (last_record_type == "CNAME" && record_type != "CNAME") ||
                (last_record_type == "NS" && record_type != "NS") ||
                (last_record_type == "TXT" && record_type != "TXT")) {
                print "";
            }
            printf "%-24s %s IN A %s\n", record_name, default_ttl, ip;
            last_record_type = "A";
        }
    }
    # Handle MX records - preserve original MX target
    else if (record_type == "MX") {
        # Add spacing between different record types (only for major type changes)
        if ((last_record_type == "A" && record_type != "A") || 
            (last_record_type == "CNAME" && record_type != "CNAME") ||
            (last_record_type == "NS" && record_type != "NS") ||
            (last_record_type == "TXT" && record_type != "TXT")) {
            print "";
        }
        # Extract the MX target from the value
        mx_target = value;
        # Remove priority if it exists in the value
        sub(/^[0-9]+ /, "", mx_target);
        # Ensure target ends with a dot
        if (mx_target !~ /\.$/) {
            mx_target = mx_target ".";
        }
        printf "%-24s %s IN MX %s %s\n", domain ".", default_ttl, mx_priority, mx_target;
        last_record_type = "MX";

    }
    # NS: deduplicate by owner+target. TXT and CNAME: keep as-is.
    else if (record_type == "NS") {
        owner = record_name;
        if (owner == "@") owner = domain ".";
        target = value;
        # Ensure target ends with a dot
        if (target !~ /\.$/) {
            target = target ".";
        }
        key = owner "|" target;
        if (!(key in ns_seen)) {
            # Add spacing between different record types (only for major type changes)
            if ((last_record_type == "A" && record_type != "A") || 
                (last_record_type == "MX" && record_type != "MX") ||
                (last_record_type == "CNAME" && record_type != "CNAME")) {
                print "";
            }
            printf "%-24s %s IN NS %s\n", owner, default_ttl, target;
            last_record_type = "NS";
            ns_seen[key] = 1;
        }
    }
    else if (record_type == "TXT" || record_type == "CNAME") {
        # Add spacing between different record types (only for major type changes)
        if ((last_record_type == "A" && record_type != "A") || 
            (last_record_type == "MX" && record_type != "MX") ||
            (last_record_type == "CNAME" && record_type != "CNAME")) {
            print "";
        }
        printf "%-24s %s IN %s %s\n", record_name, default_ttl, record_type, value;
        last_record_type = record_type;
    }
}

# Add appropriate spacing between different record types
END {
    print ""
}
' "$DA_FILE" >> "$TMP_FILE"

mv "$TMP_FILE" "$DA_FILE"

echo "Conversion complete."
echo "File $DA_FILE was updated in-place."