#!/bin/bash
# bash <( curl -s 'https://bash.license.icu/dns/cpanelsync/install-dns-sync.sh' )
# Define ANSI color codes
GREEN='\033[0;32m'
RED='\033[0;31m'
NC='\033[0m' # No Color

# Detect operating system - Linux only
if [ ! -f /etc/os-release ]; then
    echo -e "${RED}This script only supports Linux systems${NC}"
    exit 1
fi
OS="linux"

# Cross-distro installer for inotify-tools (provides inotifywait)
install_inotify_tools() {
    if command -v inotifywait >/dev/null 2>&1; then
        return 0
    fi

    echo "Installing inotify-tools for real-time monitoring..."

    if command -v apt-get >/dev/null 2>&1; then
        apt-get update && apt-get install -y inotify-tools || true
    elif command -v dnf >/dev/null 2>&1; then
        dnf -y install epel-release || true
        dnf -y install inotify-tools || true
    elif command -v yum >/dev/null 2>&1; then
        yum -y install epel-release || true
        yum -y install inotify-tools || true
    elif command -v zypper >/dev/null 2>&1; then
        zypper refresh || true
        zypper install -y inotify-tools || true
    elif command -v apk >/dev/null 2>&1; then
        apk update || true
        apk add inotify-tools || true
    else
        echo -e "${RED}No supported package manager found. Please install 'inotify-tools' manually.${NC}"
        return 1
    fi

    if ! command -v inotifywait >/dev/null 2>&1; then
        echo -e "${RED}Failed to install inotify-tools automatically. On RHEL/CentOS/Alma/Rocky you may need to enable EPEL first (e.g., 'yum install -y epel-release' or 'dnf install -y epel-release'), then install 'inotify-tools'.${NC}"
        return 1
    fi
}

# Function to uninstall all components
uninstall_components() {
    echo "Uninstalling Yatosha DNS components..."
    
    # Remove cron files
    rm -rf /etc/cron.d/rebuild-dns-on-newfiles
    rm -rf /etc/cron.d/yatosha-dns

    # Remove script directories
    rm -rf /home/yatosha-dns-scripts
    rm -rf /opt/yatosha-dns

    # Remove timestamp file
    rm -rf /var/run/rebuild-dns-config.time
    
    # Remove systemd service and log files
    systemctl stop yatosha-dns-monitor.service 2>/dev/null || true
    systemctl disable yatosha-dns-monitor.service 2>/dev/null || true
    rm -rf /etc/systemd/system/yatosha-dns-monitor.service
    rm -rf /var/log/yatosha-dns.log
    systemctl daemon-reload

    echo -e "${GREEN}Uninstallation complete. All components removed.${NC}"
    exit 0
}

# Check for uninstall flag
if [ "$1" = "--uninstall" ] || [ "$1" = "-u" ]; then
    uninstall_components
fi

# If no arguments provided, show menu
if [ $# -eq 0 ]; then
    echo ""
    echo "Yatosha DNS Management"
    echo "====================="
    echo "1. Install with Cron (every 2 minutes)"
    echo "2. Install with Daemon (real-time monitoring)"
    echo "3. Uninstall"
    echo ""
    read -p "Please choose an option (1-3): " choice
    
    case $choice in
        1)
            MONITORING_MODE="cron"
            ;;
        2)
            MONITORING_MODE="daemon"
            ;;
        3)
            uninstall_components
            ;;
        *)
            echo -e "${RED}Invalid option. Exiting.${NC}"
            exit 1
            ;;
    esac
fi

# Check if the /var/cpanel/dnsonly file exists
if [ -f /var/cpanel/dnsonly ]; then
    # Remove old cron file
    rm -rf /etc/cron.d/rebuild-dns-on-newfiles
    rm -rf /etc/cron.d/yatosha-dns

    # Remove old script directory
    rm -rf /home/yatosha-dns-scripts
    rm -rf /opt/yatosha-dns

    # Create the script directory
    mkdir -p /opt/yatosha-dns

    # Download the rebuild-dns-on-newfiles.sh script content from the URL
    curl -o /opt/yatosha-dns/rebuild-dns-on-newfiles.sh https://bash.license.icu/dns/cpanel-only/rebuild-dns-on-newfiles.sh

    # Make the script executable
    chmod +x /opt/yatosha-dns/rebuild-dns-on-newfiles.sh

    # Setup monitoring based on chosen mode
    if [ "$MONITORING_MODE" = "cron" ]; then
        # Create the cron file
        echo 'PATH=/usr/local/sbin:/usr/sbin:/sbin:/usr/local/bin:/usr/bin:/bin' > /etc/cron.d/yatosha-dns
        echo '*/2 * * * * root /opt/yatosha-dns/rebuild-dns-on-newfiles.sh' > /etc/cron.d/yatosha-dns
        echo '*/2 * * * * root chown -R named:named /var/named/' >> /etc/cron.d/yatosha-dns
        
        echo ""
        echo -e "${GREEN}Cron-based monitoring installed (checks every 2 minutes)${NC}"
        
    elif [ "$MONITORING_MODE" = "daemon" ]; then
        # Ensure inotify-tools is installed (for inotifywait)
        install_inotify_tools || { echo -e "${RED}inotify-tools is required for daemon mode. Aborting.${NC}"; exit 1; }
        
        # Create systemd service for Linux daemon
        cat > /etc/systemd/system/yatosha-dns-monitor.service << 'EOF'
[Unit]
Description=Yatosha DNS Monitor Service
After=network.target

[Service]
Type=simple
ExecStart=/opt/yatosha-dns/rebuild-dns-on-newfiles.sh --daemon
Restart=always
RestartSec=5
User=root
StandardOutput=file:/var/log/yatosha-dns.log
StandardError=file:/var/log/yatosha-dns.log

[Install]
WantedBy=multi-user.target
EOF
        
        # Reload systemd and start the service
        systemctl daemon-reload
        systemctl enable yatosha-dns-monitor.service
        systemctl start yatosha-dns-monitor.service
        
        echo ""
        echo -e "${GREEN}Real-time daemon monitoring installed${NC}"
        echo -e "${GREEN}Logs available at: /var/log/yatosha-dns.log${NC}"
    fi

    echo ""
    echo "To uninstall, run: $0 --uninstall"
else
    echo ""
    echo -e "${RED}/var/cpanel/dnsonly file not found. The script will not run.${NC}"
fi
