#!/bin/sh

# POSIX sh-compatible DNS rebuild script
# - Daemon mode uses inotifywait (from inotify-tools)
# - Batch mode (cron/manual) scans /var/named for .db files changed since last run

# Configurable batching parameters (can be overridden via env)
THRESHOLD="${YATOSHA_BATCH_THRESHOLD:-10}"
DEBOUNCE="${YATOSHA_DEBOUNCE_SECONDS:-5}"
CHANGES_FILE="/var/run/yatosha-dns.changes"
SCHEDULE_FILE="/var/run/yatosha-dns.scheduled"
LOCK_DIR="/var/run/yatosha-dns.lock"

# Helper: ensure domain exists in /etc/userdomains mapped to 'nobody'
ensure_userdomains_entry() {
    d="$1"
    [ -n "$d" ] || return 0
    file="/etc/userdomains"
    # Create file if it doesn't exist
    if [ ! -f "$file" ]; then
        : > "$file" 2>/dev/null || return 0
    fi
    # Avoid duplicates; match exact 'domain:' prefix
    if ! grep -Fq "$d:" "$file" 2>/dev/null; then
        printf '%s: nobody\n' "$d" >> "$file"
    fi
}

# Helper: remove domain entry from /etc/userdomains
remove_userdomains_entry() {
    d="$1"
    [ -n "$d" ] || return 0
    file="/etc/userdomains"
    [ -f "$file" ] || return 0
    tmp=$(mktemp /tmp/userdomains-XXXXXX 2>/dev/null) || return 0
    # Keep lines that do not start with "<domain>:"
    grep -Fv "^$d:" "$file" > "$tmp" 2>/dev/null && mv "$tmp" "$file" 2>/dev/null || rm -f "$tmp"
}

# Helper: ensure zone file owner is 'named'
ensure_owner_named() {
    f="$1"
    [ -n "$f" ] || return 0
    chown named:named "$f" 2>/dev/null || chown named "$f" 2>/dev/null || true
}

# Helper: reload a single zone via PowerDNS bind backend
reload_zone() {
    d="$1"
    [ -n "$d" ] || return 0
    if [ -x "/usr/bin/pdns_control" ]; then
        /usr/bin/pdns_control bind-reload-now "$d" >/dev/null 2>&1 || /usr/bin/pdns_control bind-reload-now "$d"
    else
        echo "pdns_control not found at /usr/bin/pdns_control; skipping reload for $d"
    fi
}

if [ "${1-}" = "--daemon" ]; then
    echo "Starting real-time DNS configuration monitor..."

    # Ensure inotifywait is available (installer should have installed inotify-tools)
    if ! command -v inotifywait >/dev/null 2>&1; then
        echo "inotifywait not found. Please install 'inotify-tools' first (handled by install-cpanel.sh)."
        exit 1
    fi

    # Batch processor for queued file changes
    process_changes() {
        # Debounce window to accumulate rapid changes
        sleep "$DEBOUNCE"

        # Acquire lock (mkdir succeeds only if lock doesn't exist)
        if ! mkdir "$LOCK_DIR" 2>/dev/null; then
            # Another processor is running; defer
            return 0
        fi

        # Ensure unique list if any queued
        COUNT=0
        if [ -f "$CHANGES_FILE" ]; then
            sort -u "$CHANGES_FILE" 2>/dev/null > "$CHANGES_FILE.tmp" || true
            mv "$CHANGES_FILE.tmp" "$CHANGES_FILE" 2>/dev/null || true
            COUNT=$(wc -l < "$CHANGES_FILE" 2>/dev/null | tr -d ' ')
            [ -n "$COUNT" ] || COUNT=0
        fi

        if [ "$COUNT" -eq 0 ] 2>/dev/null; then
            rm -f "$SCHEDULE_FILE"
            rmdir "$LOCK_DIR" 2>/dev/null || true
            return 0
        fi

        if [ "$COUNT" -ge "$THRESHOLD" ] 2>/dev/null; then
            echo "Many changes detected ($COUNT >= $THRESHOLD). Converting all zones and reloading individually..."
            find /var/named -type f -name '*.db' -print 2>/dev/null |
            while IFS= read -r f; do
                [ -n "$f" ] || continue
                [ -f "$f" ] || continue
                /opt/yatosha-dns/convert_da_to_cpanel.sh "$f"
                base=$(basename "$f"); d=${base%.db}; ensure_userdomains_entry "$d"; ensure_owner_named "$f"; reload_zone "$d"
            done
        else
            echo "Processing $COUNT changed zone file(s) before reload..."
            while IFS= read -r f; do
                [ -n "$f" ] || continue
                [ -f "$f" ] || continue
                /opt/yatosha-dns/convert_da_to_cpanel.sh "$f"
                base=$(basename "$f"); d=${base%.db}; ensure_userdomains_entry "$d"; ensure_owner_named "$f"; reload_zone "$d"
            done < "$CHANGES_FILE"
        fi

        # Update timestamp for last processing
        date +%s > /var/run/rebuild-dns-config.time
        echo "Zone processing and reload completed"

        # Clear queue and release lock
        : > "$CHANGES_FILE"
        rm -f "$SCHEDULE_FILE"
        rmdir "$LOCK_DIR" 2>/dev/null || true
    }

    # Monitor /var/named directory for .db file changes (include event type)
    inotifywait -m -r -e create,modify,delete,move /var/named --format '%w%f|%e' 2>/dev/null |
    while IFS= read -r LINE; do
        FILE=$(printf '%s' "$LINE" | awk -F'|' '{print $1}')
        EVENTS=$(printf '%s' "$LINE" | awk -F'|' '{print $2}')
        case "$FILE" in
            *.db)
                case "$FILE" in *.swp|*.tmp) continue ;; esac
                base=$(basename "$FILE"); d=${base%.db}
                case "$EVENTS" in
                    *DELETE*|*MOVED_FROM*)
                        # On deletion, remove from /etc/userdomains
                        remove_userdomains_entry "$d"
                        ;;
                    *)
                        # Queue the file for processing on create/modify/move-in
                        printf '%s\n' "$FILE" >> "$CHANGES_FILE"
                        # Schedule a batch processor if not already scheduled
                        if [ ! -f "$SCHEDULE_FILE" ]; then
                            : > "$SCHEDULE_FILE"
                            process_changes &
                        fi
                        ;;
                esac
                ;;
        esac
    done
else
    # Batch mode (cron/manual)
    last_time=""
    if [ -f /var/run/rebuild-dns-config.time ]; then
        last_time=$(cat /var/run/rebuild-dns-config.time 2>/dev/null || echo "")
    fi

    tmpfile=$(mktemp /tmp/rebuild-dns-XXXXXX) || exit 1

    if [ -n "$last_time" ]; then
        # Collect only files modified since last_time
        find /var/named -type f -name '*.db' -print 2>/dev/null |
        while IFS= read -r file; do
            [ -n "$file" ] || continue
            ft=$(stat -c %Y "$file" 2>/dev/null || echo 0)
            # Compare numerically; guard against empty/invalid values
            if [ "$ft" -gt "$last_time" ] 2>/dev/null; then
                printf '%s\n' "$file" >> "$tmpfile"
            fi
        done
    else
        # First run: process all .db files
        find /var/named -type f -name '*.db' -print 2>/dev/null > "$tmpfile"
    fi

    if [ -s "$tmpfile" ]; then
        # Convert all modified .db files to cPanel format and reload each zone
        while IFS= read -r file; do
            [ -n "$file" ] || continue
            [ -f "$file" ] || continue
            /opt/yatosha-dns/convert_da_to_cpanel.sh "$file"
            base=$(basename "$file"); d=${base%.db}; ensure_userdomains_entry "$d"; ensure_owner_named "$file"; reload_zone "$d"
        done < "$tmpfile"

        # Update timestamp with current time
        date +%s > /var/run/rebuild-dns-config.time
        echo "Zone processing and reload completed"
    fi

    rm -f "$tmpfile"
fi