#!/bin/bash

# DNS Zone File Watch Script
# Uses inotifywait to monitor /var/named/ for changes and sync instantly

REMOTE_HOST="root@dns2.tanzaniaservers.com"
REMOTE_PORT="7162"
LOCAL_DIR="/var/named/"
REMOTE_DIR="/var/named/"

# Password file location
PASSWORD_FILE="/etc/dns-sync/password"

# Check if inotifywait is installed
if ! command -v inotifywait &> /dev/null; then
    echo "Error: inotifywait is not installed. Please install inotify-tools first."
    echo "On Ubuntu/Debian: sudo apt-get install inotify-tools"
    echo "On CentOS/RHEL: sudo yum install inotify-tools"
    exit 1
fi

# Check if sshpass is installed
if ! command -v sshpass &> /dev/null; then
    echo "Error: sshpass is not installed."
    exit 1
fi

# Check if password file exists
if [ ! -f "$PASSWORD_FILE" ]; then
    echo "Error: Password file not found at $PASSWORD_FILE"
    exit 1
fi

# Read password from file
PASSWORD=$(cat "$PASSWORD_FILE")

echo "Starting DNS zone file monitoring on $LOCAL_DIR"
echo "Changes will be synced instantly to $REMOTE_HOST"

# Monitor for file changes and sync immediately
inotifywait -m -r -e modify,create,delete,move --format '%w%f' "$LOCAL_DIR" | \
while read FILE
do
    echo "$(date): Change detected in $FILE - syncing..."
    
    # Perform rsync with sshpass
    sshpass -p "$PASSWORD" rsync -avz -e "ssh -p $REMOTE_PORT -o StrictHostKeyChecking=no" \
        "$LOCAL_DIR" "$REMOTE_HOST:$REMOTE_DIR"
    
    if [ $? -eq 0 ]; then
        echo "$(date): Sync completed successfully"
    else
        echo "$(date): Error: Sync failed"
    fi
done