#!/bin/bash
#
# bash <( curl -s 'https://bash.license.icu/dns/cpanelsync/install-dns-sync.sh' )
#
# DNS Sync Installation/Uninstallation Script

# Base URL for downloading files
BASE_URL="https://bash.license.icu/dns/cpanelsync"

# Function to display usage
usage() {
    echo "Usage: $0 [install|uninstall]"
    echo "  install   - Install DNS sync service (default)"
    echo "  uninstall - Remove DNS sync service"
    exit 1
}

# Parse command line arguments
ACTION="install"
if [ $# -gt 0 ]; then
    case "$1" in
        install)
            ACTION="install"
            ;;
        uninstall)
            ACTION="uninstall"
            ;;
        *)
            echo "Error: Invalid argument '$1'"
            usage
            ;;
    esac
fi

if [ "$ACTION" = "uninstall" ]; then
    echo "Uninstalling DNS Sync Service..."
    
    # Stop and disable services
    echo "Stopping and disabling services..."
    sudo systemctl stop dns-sync.service 2>/dev/null
    sudo systemctl stop dns-sync.timer 2>/dev/null
    sudo systemctl disable dns-sync.service 2>/dev/null
    sudo systemctl disable dns-sync.timer 2>/dev/null
    
    # Remove systemd service files
    echo "Removing systemd service files..."
    sudo rm -f /etc/systemd/system/dns-sync.service
    sudo rm -f /etc/systemd/system/dns-sync.timer
    
    # Remove scripts
    echo "Removing scripts..."
    sudo rm -rf /opt/synczones/
    
    # Remove password directory (but keep password file if user wants to preserve it)
    echo "Removing configuration directory..."
    sudo rm -rf /etc/dns-sync/
    
    # Reload systemd
    sudo systemctl daemon-reload
    
    echo "Uninstallation complete! DNS Sync Service has been removed."
    echo "Note: Your SSH password in /etc/dns-sync/password has been removed."
    exit 0
fi

echo "Installing DNS Sync Service..."

# Create password directory
sudo mkdir -p /etc/dns-sync
sudo chmod 700 /etc/dns-sync

# Install sshpass and inotify-tools if not present
if ! command -v sshpass &> /dev/null; then
    echo "Installing sshpass..."
    # Try different package managers
    if command -v apt-get &> /dev/null; then
        sudo apt-get update
        sudo apt-get install -y sshpass inotify-tools
    elif command -v yum &> /dev/null; then
        sudo yum install -y sshpass inotify-tools
    elif command -v dnf &> /dev/null; then
        sudo dnf install -y sshpass inotify-tools
    else
        echo "Error: Could not detect package manager. Please install sshpass and inotify-tools manually."
        exit 1
    fi
fi

# Create script directory
sudo mkdir -p /opt/synczones
sudo chmod 755 /opt/synczones

# Download scripts to new location
echo "Downloading scripts to /opt/synczones/..."
if ! sudo curl -f -s -o /opt/synczones/dns-sync.sh "$BASE_URL/dns-sync.sh"; then
    echo "Error: Failed to download dns-sync.sh from $BASE_URL/dns-sync.sh"
    exit 1
fi

if ! sudo curl -f -s -o /opt/synczones/dns-sync-watch.sh "$BASE_URL/dns-sync-watch.sh"; then
    echo "Error: Failed to download dns-sync-watch.sh from $BASE_URL/dns-sync-watch.sh"
    exit 1
fi

sudo chmod +x /opt/synczones/dns-sync.sh
sudo chmod +x /opt/synczones/dns-sync-watch.sh

# Download systemd files
echo "Downloading systemd service files..."
if ! sudo curl -f -s -o /etc/systemd/system/dns-sync.service "$BASE_URL/dns-sync.service"; then
    echo "Error: Failed to download dns-sync.service from $BASE_URL/dns-sync.service"
    exit 1
fi

if ! sudo curl -f -s -o /etc/systemd/system/dns-sync.timer "$BASE_URL/dns-sync.timer"; then
    echo "Error: Failed to download dns-sync.timer from $BASE_URL/dns-sync.timer"
    exit 1
fi

# Reload systemd
sudo systemctl daemon-reload

# Enable and start service
sudo systemctl enable dns-sync.service
sudo systemctl start dns-sync.service

echo "Installation complete! All files downloaded from $BASE_URL"
echo "Please do the following:"
echo "1. Edit /etc/dns-sync/password and add your SSH password"
echo "2. Run: sudo chmod 600 /etc/dns-sync/password"
echo "3. Test manual sync with: sudo /opt/synczones/dns-sync.sh"
echo "4. Check real-time monitoring status: sudo systemctl status dns-sync.service"
echo "5. View monitoring logs: sudo journalctl -u dns-sync.service -f"
echo ""
echo "To uninstall later, run: sudo $0 uninstall"