#!/bin/bash

# Custom DirectAdmin post-hook for DNS writes
# - Sources central config for settings (LOGFILE, DEBUG, time)
# - Triggers zone synchronization to remote servers
# - Critically: never return a non-zero exit code to DirectAdmin to avoid interrupting DA's own serial handling

# Source config if present; otherwise use safe defaults
if [ -r /opt/yatosha-dns/config.sh ]; then
  source /opt/yatosha-dns/config.sh >/dev/null 2>&1 || true
else
  LOGFILE="/var/log/yatosha-dns.log"
  DEBUG=false
  time=0
fi

# Get domain and IP from DirectAdmin environment variables
# DA provides DOMAIN and DOMAIN_IP during dns_write hooks
# We read from env for robustness across DA versions

da_domain=$(env | awk -F= '/^DOMAIN=/{print $2; exit}')
da_ip=$(env | awk -F= '/^DOMAIN_IP=/{print $2; exit}')

# Validate required variables
# Important: Do NOT fail the hook; log and exit 0 so DA can continue its own work (including serial updates)
if [ -z "$da_domain" ] || [ -z "$da_ip" ]; then
    $DEBUG && echo "$(date '+%b %e %T') ERROR: Missing DOMAIN or DOMAIN_IP environment variables" >> $LOGFILE
    exit 0
fi

# Filter out system domains and unwanted domains
case "$da_domain" in
    localhost|*.local|*.localdomain|*.internal)
        $DEBUG && echo "$(date '+%b %e %T') SKIP: System domain $da_domain" >> $LOGFILE
        exit 0
        ;;
    *)
        # Process regular domains
        ;;
esac

# Execute DNS sync immediately for DirectAdmin feedback
if [ "$time" -eq 0 ]; then
    # Immediate execution
    $DEBUG && echo "$(date '+%b %e %T') EXECUTING: DNS sync for $da_domain ($da_ip)" >> $LOGFILE
    # Redirect all output of the sync script to the log to avoid surfacing to DirectAdmin UI
    /opt/yatosha-dns/yatosha-dns.sh sync "${da_domain}" --ipaddr "${da_ip}" >> "$LOGFILE" 2>&1
    exit_code=$?
    if [ $exit_code -eq 0 ]; then
        $DEBUG && echo "$(date '+%b %e %T') SUCCESS: DNS sync for $da_domain ($da_ip)" >> $LOGFILE
    else
        $DEBUG && echo "$(date '+%b %e %T') FAILED: DNS sync for $da_domain ($da_ip) with exit code $exit_code" >> $LOGFILE
    fi
    # Never propagate failure back to DirectAdmin; DA must be able to write and bump serial independently
    exit 0
else
    # Delayed execution using at command
    task_file="/tmp/yatosha-dns-write-$$-$(date +%s).task"
    # Ensure the job writes its own output to the central log file; avoid relying on environment inside 'at'
    echo "/opt/yatosha-dns/yatosha-dns.sh sync ${da_domain} --ipaddr ${da_ip} >> /var/log/yatosha-dns.log 2>&1" > "$task_file"
    
    # Schedule the task
    /usr/bin/at now + $time minutes < "$task_file" >/dev/null 2>&1
    
    # Clean up task file
    rm -f "$task_file"
    
    $DEBUG && echo "$(date '+%b %e %T') QUEUED: DNS sync for $da_domain ($da_ip)" >> $LOGFILE
fi
