#!/bin/bash
usage() {
    echo "Usage   : yatosha-dns.sh COMMAND DOMAIN [--nshost] [--ipaddr]"
    echo "Example : "
    echo "   yatosha-dns.sh sync example.com --nshost ns1 --ipaddr 192.168.1.1"
    echo "   yatosha-dns.sh delete example.com"
    echo
    exit 0
}

source /opt/yatosha-dns/config.sh

now() {
    echo $(date +"%b %e %T")
}

bash_run() {
    local output=$( bash <<EOF
$1
EOF
)
    echo "$output"
}

sync_dns() {
    host=$1
    pass=$2
    domain=$3
    ip=$4
    local sync_error=0
    # Prepare SSH multiplexing options for fewer handshakes and faster reuse
    SSH_OPTS="-oStrictHostKeyChecking=no -p ${port} -o ControlMaster=auto -o ControlPath=/tmp/yatosha_dns_cm_%r@%h:%p -o ControlPersist=60s -o ConnectTimeout=5"
    
    # Verify local zone file exists and is writable
    if [ ! -f "/var/named/${domain}.db" ]; then
        echo "ERROR: Zone file /var/named/${domain}.db not found" >&2
        $DEBUG && echo "$(now) ERROR: Zone file /var/named/${domain}.db not found" >> $LOGFILE
        return 1
    fi
    
    if [ ! -w "/var/named/${domain}.db" ]; then
        echo "ERROR: Zone file /var/named/${domain}.db is not writable" >&2
        $DEBUG && echo "$(now) ERROR: Zone file /var/named/${domain}.db is not writable" >> $LOGFILE
        return 1
    fi
    
    # Prefer the first IPv4 address that isn't 176.99.3.34
    server_ip=$(hostname -I 2>/dev/null | tr ' ' '\n' | awk '$1 != "176.99.3.34" && $1 ~ /^([0-9]{1,3}\.){3}[0-9]{1,3}$/ {print $1; exit}')
    
    # Replace the old server IP only in this domain's zone file, and only if it contains the old IP
    if [ -n "$server_ip" ] && grep -q "176\\.99\\.3\\.34" "/var/named/${domain}.db" 2>/dev/null; then
        # Escape sed replacement special chars: &, /, and \
        safe_server_ip=${server_ip//\\/\\\\}
        safe_server_ip=${safe_server_ip//&/\\&}
        safe_server_ip=${safe_server_ip//\//\\/}
        sed -i "s/176\\.99\\.3\\.34/${safe_server_ip}/g" "/var/named/${domain}.db"
    fi
    
    # Ensure proper ownership/mode only if they are incorrect (avoid unnecessary chown/chmod)
    current_owner=$(stat -c '%U:%G' "/var/named/${domain}.db" 2>/dev/null || stat -f '%Su:%Sg' "/var/named/${domain}.db")
    if [ "$current_owner" != "named:named" ]; then
        chown named:named "/var/named/${domain}.db"
    fi
    # Get numeric mode; GNU stat first, BSD fallback extracts last 3 digits
    current_mode=$(stat -c '%a' "/var/named/${domain}.db" 2>/dev/null || (stat -f '%p' "/var/named/${domain}.db" | sed 's/^.*\(.\{3\}\)$/\1/'))
    if [ "$current_mode" != "600" ]; then
        chmod 600 "/var/named/${domain}.db"
    fi
    
    # SSH connectivity and remote writability will be validated implicitly via rsync/SSH commands below
    
    # Check if zone exists on remote server (cPanel) and create if missing
    if ! sshpass -p "${pass}" ssh ${SSH_OPTS} root@${host} "whmapi1 dumpzone domain=${domain}" 2>/dev/null | grep -q 'result: 1'; then
        # Zone doesn't exist, create it
        if ! sshpass -p "${pass}" ssh ${SSH_OPTS} root@${host} "whmapi1 adddns domain=${domain} ip=${ip}" >/dev/null 2>&1; then
            echo "ERROR: Failed to create DNS zone ${domain} on ${host}" >&2
            $DEBUG && echo "$(now) ERROR: Failed to create DNS zone ${domain} on ${host}" >> $LOGFILE
            sync_error=1
        else
            $DEBUG && echo "$(now) Created DNS zone ${domain} on ${host}" >> $LOGFILE
        fi
    else 
        $DEBUG && echo "$(now) Zone already exists ${domain} on ${host}" >> $LOGFILE
    fi

    # Sync with detailed error output (no compression for tiny files)
    rsync_output=$(rsync -a --chown=named:named --chmod=600 -e "sshpass -p '${pass}' ssh ${SSH_OPTS}" /var/named/${domain}.db root@${host}:/var/named/ 2>&1)
    if [ $? -ne 0 ]; then
        echo "ERROR: Failed to sync zone file ${domain}.db to ${host}" >&2
        echo "Rsync error: ${rsync_output}" >&2
        $DEBUG && echo "$(now) ERROR: Failed to sync zone file ${domain}.db to ${host}. Error: ${rsync_output}" >> $LOGFILE
        sync_error=1
    else
        $DEBUG && echo "$(now) Successfully synced zone file ${domain}.db to ${host}" >> $LOGFILE
    fi

    # Reload the zone on the remote server using pdns_control (no logging)
    if sshpass -p "${pass}" ssh ${SSH_OPTS} root@${host} "sudo /usr/bin/pdns_control bind-reload-now ${domain}" >/dev/null 2>&1; then
        true # No logging for this action
    else
        echo "ERROR: Failed to reload zone ${domain} on ${host}" >&2
        sync_error=1
    fi
}

delete_dns() {
    host=$1
    pass=$2
    domain=$3
    local delete_error=0
    # Reuse SSH multiplexing for deletes as well
    SSH_OPTS="-oStrictHostKeyChecking=no -p ${port} -o ControlMaster=auto -o ControlPath=/tmp/yatosha_dns_cm_%r@%h:%p -o ControlPersist=60s -o ConnectTimeout=5"

    # Delete DNS zone on remote server
    if ! sshpass -p "${pass}" ssh ${SSH_OPTS} root@${host} "whmapi1 killdns domain=${domain}" >/dev/null 2>&1; then
        echo "ERROR: Failed to delete DNS zone ${domain} on ${host}" >&2
        $DEBUG && echo "$(now) ERROR: Failed to delete DNS zone ${domain} on ${host}" >> $LOGFILE
        delete_error=1
    else
        $DEBUG && echo "$(now) Deleted DNS zone ${domain} on ${host}" >> $LOGFILE
    fi

    # Also ensure the domain is removed from /etc/userdomains on the remote server (cPanel)
    # Escape dots for regex so sed matches the exact domain prefix followed by colon
    safe_domain_regex=${domain//./\\.}
    if sshpass -p "${pass}" ssh ${SSH_OPTS} root@${host} "if [ -f /etc/userdomains ]; then sed -i \"/^${safe_domain_regex}:/d\" /etc/userdomains; fi" >/dev/null 2>&1; then
        $DEBUG && echo "$(now) Ensured ${domain} removed from /etc/userdomains on ${host}" >> $LOGFILE
    else
        $DEBUG && echo "$(now) INFO: Could not update /etc/userdomains for ${domain} on ${host} (file missing or permission issue)" >> $LOGFILE
    fi
    
    return $delete_error
}

ipaddr=$( head -1 /usr/local/directadmin/data/admin/ip.list ) #server ip address
nshost="all"

POSITIONAL=()
while [[ $# -gt 0 ]]
do
    key="$1"
    case $key in
        --nshost)
            nshost="$2"
            shift
            shift
            ;;
        --ipaddr)
            ipaddr="$2"
            shift
            shift
            ;;
        --default)
            DEFAULT=YES
            shift
            ;;
        *)   
            POSITIONAL+=("$1") 
            shift
            ;;
    esac
done
set -- "${POSITIONAL[@]}"

if [ -z "$1" ] || [ $1 = "-h" ] || [ -z "$2" ] ; then
    usage
fi

cmd=$1
domain=$2

overall_error=0

if [ $cmd = "sync" ] ; then
    if [ $nshost = "all" ] ; then
        for ns in "${!host[@]}" ; do
            if ! sync_dns ${host[$ns]} ${pass[$ns]} $domain $ipaddr; then
                overall_error=1
            fi
        done
    else
        if ! sync_dns ${host[$nshost]} ${pass[$nshost]} $domain $ipaddr; then
            overall_error=1
        fi
    fi
fi

if [ $cmd = "delete" ] ; then
    if [ $nshost = "all" ] ; then
        for ns in "${!host[@]}" ; do
            if ! delete_dns ${host[$ns]} ${pass[$ns]} $domain; then
                overall_error=1
            fi
        done
    else
        if ! delete_dns ${host[$nshost]} ${pass[$nshost]} $domain; then
            overall_error=1
        fi
    fi
fi

exit $overall_error
